<?php

/**
 *
 *    Copyright (C) 2017 onOffice GmbH
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU Affero General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU Affero General Public License for more details.
 *
 *    You should have received a copy of the GNU Affero General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

namespace onOffice\WPlugin\Gui\Table;

use onOffice\SDK\onOfficeSDK;
use onOffice\WPlugin\API\APIClientCredentialsException;
use onOffice\WPlugin\Controller\Exception\UnknownFilterException;
use onOffice\WPlugin\Controller\UserCapabilities;
use onOffice\WPlugin\FilterCall;
use onOffice\WPlugin\Gui\AdminPageEstateListSettingsBase;
use onOffice\WPlugin\Gui\Table\WP\ListTable;
use onOffice\WPlugin\Model\FormModelBuilder\FormModelBuilderDBEstateListSettings;
use onOffice\WPlugin\Record\RecordManagerReadListViewEstate;
use WP_List_Table;
use function __;
use function admin_url;
use function current_user_can;
use function esc_html;
use function esc_html__;
use function esc_js;
use function wp_nonce_url;


/**
 *
 */

class EstateListTable extends ListTable
{
	/** @var int */
	private $_itemsPerPage = null;

	/** @var FilterCall */
	private $_pFilterCall = null;

	/**
	 *
	 * @see WP_List_Table::__construct() for more information on default arguments.
	 *
	 * @param array $args An associative array of arguments.
	 *
	 */

	public function __construct($args = [])
	{
		parent::__construct([
			'singular' => 'estatelist',
			'plural' => 'estatelists',
			'screen' => $args['screen'] ?? null,
		]);

		$this->_itemsPerPage = $this->get_items_per_page('onoffice_estate_listview_per_page', 20);
		$this->_pFilterCall = new FilterCall(onOfficeSDK::MODULE_ESTATE);
	}


	/**
	 *
	 * @return bool
	 *
	 */

	public function ajax_user_can()
	{
		$pUserCapabilities = new UserCapabilities();
		$roleEditEstates = $pUserCapabilities->getCapabilityForRule
			(UserCapabilities::RULE_EDIT_VIEW_ESTATE);
		return current_user_can($roleEditEstates);
	}


	/**
	 *
	 */

	private function fillData()
	{
		$page = $this->get_pagenum() - 1;
		$itemsPerPage = $this->_itemsPerPage;
		$offset = $page * $itemsPerPage;

		$pRecordRead = new RecordManagerReadListViewEstate();
		$pRecordRead->setLimit($itemsPerPage);
		$pRecordRead->setOffset($offset);
		$pRecordRead->addColumn('listview_id', 'ID');
		$pRecordRead->addColumn('name');
		$pRecordRead->addColumn('filterId');
		$pRecordRead->addColumn('template');
		$pRecordRead->addColumn('list_type');
		$pRecordRead->addColumn('name', 'shortcode');
		$pRecordRead->addColumn('page_shortcode');
		$pRecordRead->addWhere("`list_type` IN('default', 'reference', 'favorites')");

		$pRecord = $pRecordRead->getRecordsSortedAlphabetically();
		$pRecord = $this->handleRecord($pRecord);
		$this->setItems($pRecord);
		$itemsCount = $pRecordRead->getCountOverall();

		$this->set_pagination_args([
			'total_items' => $itemsCount,
			'per_page' => $itemsPerPage,
			'total_pages' => ceil($itemsCount / $itemsPerPage),
		]);
	}

	/**
	 *
	 */

	public function prepare_items()
	{
		$columns = [
			'cb' => '<input type="checkbox" />',
			'name' => __('List name', 'onoffice-for-wp-websites'),
			'filtername' => __('Selected filter', 'onoffice-for-wp-websites'),
			'template' => __('Template', 'onoffice-for-wp-websites'),
			'list_type' => __('List type', 'onoffice-for-wp-websites'),
			'shortcode' => __('Shortcode', 'onoffice-for-wp-websites'),
			'page_shortcode' => __('Pages using the shortcode', 'onoffice-for-wp-websites'),
		];

		$hiddenColumns = get_hidden_columns($this->screen);
		$hidden = array_merge($hiddenColumns, ['ID', 'filterId']);
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = [$columns, $hidden, $sortable,
			$this->get_default_primary_column_name()];

		$this->fillData();
	}

	/**
	 *
	 */

	public function get_sortable_columns()
	{
		$columns = array(
			'name' => ['name', false],
			'template' => ['template', false],
			'list_type' => ['list_type', false],
		);
		return $columns;
	}

	/**
	 *
	 * @param object $pItem
	 * @return string|null
	 *
	 */

	protected function column_list_type($pItem)
	{
		$listTypes = FormModelBuilderDBEstateListSettings::getListViewLabels();
		$selectedType = $pItem->list_type;
		return $listTypes[$selectedType] ?? null;
	}


	/**
	 *
	 * @param object $pItem
	 * @return string
	 *
	 */

	protected function column_filtername($pItem)
	{
		$filterName = '';
		try {
			if ($pItem->filterId != 0) {
				$filterName = $this->_pFilterCall->getFilternameById($pItem->filterId);
			}
		} catch (APIClientCredentialsException $pCredentialsException) {
			$filterName = __('(Needs valid API credentials)', 'onoffice-for-wp-websites');
		} catch (UnknownFilterException $pFilterException) {
			/* translators: %s will be replaced with a number. */
			$filterName = sprintf(__('(Unknown Filter (ID: %s))', 'onoffice-for-wp-websites'),
				$pFilterException->getFilterId());
		}
		return $filterName;
	}


	/**
	 *
	 * @return array
	 *
	 */

	public function get_columns()
	{
		return [
			'cb' => '<input type="checkbox" />',
			'name' => __('Name of View', 'onoffice-for-wp-websites'),
			'filtername' => __('Filter', 'onoffice-for-wp-websites'),
			'list_type' => __('Type of List', 'onoffice-for-wp-websites'),
			'shortcode' => __('Shortcode', 'onoffice-for-wp-websites'),
		];
	}


	/**
	 *
	 * @param object $pItem
	 * @return string
	 *
	 */

	protected function column_shortcode( $pItem )
	{
		return '<input type="text" style="max-width: 100%; margin-right: 5px;" readonly value="[oo_estate view=&quot;' . esc_html( $pItem->name ) . '&quot;]"><input type="button" class="button button-copy" data-clipboard-text="[oo_estate view=&quot;' . esc_html( $pItem->name ) . '&quot;]" value="' . esc_html__( 'Copy',
				'onoffice-for-wp-websites' ) . '" ><script>if (navigator.clipboard) { jQuery(".button-copy").show(); }</script>';
	}


	/**
	 * Get the name of the default primary column.
	 *
	 * @since 4.3.0
	 *
	 * @return string Name of the default primary column
	 */

	protected function get_default_primary_column_name()
	{
		return 'name';
	}


	/**
	 * Generates and displays row action links.
	 *
	 * @param object $pItem Link being acted upon.
	 * @param string $column_name Current column name.
	 * @param string $primary Primary column name.
	 * @return string Row action output for links.
	 *
	 */

	protected function handle_row_actions($pItem, $column_name, $primary)
	{
		if ($primary !== $column_name) {
			return '';
		}

		$editLink = add_query_arg(AdminPageEstateListSettingsBase::GET_PARAM_VIEWID, $pItem->ID,
			admin_url('admin.php?page=onoffice-editlistview'));

		$actions = [];
		$actions['edit'] = '<a href="'.esc_attr($editLink).'">'.esc_html__('Edit').'</a>';
		$actions['duplicate'] = "<a class='button-duplicate' href='"
			. esc_attr(wp_nonce_url(admin_url('admin.php') . '?page=onoffice-estates&action=bulk_duplicate&listVewId=' . $pItem->ID,
				'bulk-estatelists'))
			. "'>" . esc_html__('Duplicate', 'onoffice-for-wp-websites') . "</a>";
		$actions['delete'] = "<a class='submitdelete' href='"
			. esc_attr(wp_nonce_url(admin_url('admin.php').'?page=onoffice-estates&action=bulk_delete&estatelist[]='.$pItem->ID, 'bulk-estatelists'))
			."' onclick=\"if ( confirm( '"
			.esc_js(sprintf(
			/* translators: %s is the name of the list view. */
			__("You are about to delete the listview '%s'\n  'Cancel' to stop, 'OK' to delete.", 'onoffice-for-wp-websites'), $pItem->name))
			."' ) ) { return true;}return false;\">" . esc_html__('Delete') . "</a>";
		return $this->row_actions($actions);
	}
}
